package pt.caf_pbs.schooldiskapi.controller;

import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.server.ResponseStatusException;
import pt.caf_pbs.schooldiskapi.dao.FilesOperation;
import pt.caf_pbs.schooldiskapi.dto.EnrollCreateDTO;
import pt.caf_pbs.schooldiskapi.dto.EnrollEditDTO;
import pt.caf_pbs.schooldiskapi.exceptions.DuplicatedDataException;
import pt.caf_pbs.schooldiskapi.model.*;

import java.util.List;

@RestController
@RequestMapping("/v1/enroll")
public class EnrollsController {

    @PostMapping()
    public Enroll createEnroll(@RequestBody EnrollCreateDTO enrollDTO) {
        School school = FilesOperation.loadSchool();
        try {
            Student student = school.getStudentContainer().get(enrollDTO.getStudentNumber());
            if(student == null) {
                throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Student with id " + enrollDTO.getStudentNumber() + " found");
            }
            Subject subject = school.getSubjectContainer().get(enrollDTO.getSubjectInitials());
            if(subject == null) {
                throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No subject initials " + enrollDTO.getSubjectInitials() + " found");
            }

            Enroll enroll = new Enroll(enrollDTO.getGrade(), subject, student);
            school.getEnrollContainer().add(enroll);
            FilesOperation.storeSchool(school);
            return enroll;
        }catch(DuplicatedDataException e){
            throw new ResponseStatusException(HttpStatus.CONFLICT, "Problems with "+e.getMessage(), e);
        }
    }

    @GetMapping( path = "{studentId}/subjects")
    public List<Pair<Subject, Integer>> getStudentSubjects(@PathVariable Integer studentId) {
        School school = FilesOperation.loadSchool();
        Student student = school.getStudentContainer().get(studentId);
        if(student != null){
            List<Pair<Subject, Integer>> enrolls = school.getStudentContainer().getEnrolls().getSubjects(studentId);
            if(enrolls != null) {
                return enrolls;
            }else{
                throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Enrolls with studentId" + studentId + " found");
            }
        }else{
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Student with id " + studentId + " found");
        }
    }

    @GetMapping( path = "{studentId}/subject/{subjectInitials}")
    public Enroll getStudentSubject(@PathVariable Integer studentId, @PathVariable String subjectInitials) {
        School school = FilesOperation.loadSchool();
        Student student = school.getStudentContainer().get(studentId);
        if(student != null){
            Enroll enroll = school.getStudentContainer().getEnrolls().get(studentId, subjectInitials);
            if(enroll != null) {
                return enroll;
            }else{
                throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Enroll with studentId" + studentId + "  and subject initials " + subjectInitials + " found");
            }
        }else{
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Student with id " + studentId + " found");
        }
    }

    @PutMapping( path = "{studentId}/subject/{subjectInitials}")
    public void updateLecture(@PathVariable Integer studentId, @PathVariable String subjectInitials, @RequestBody EnrollEditDTO enrollDTO) {
        School school = FilesOperation.loadSchool();
        try {
            school.getStudentContainer().getEnrolls().update(studentId, subjectInitials, enrollDTO.getGrade()); //TODO: o update devolve nulo não era melhor atirar exceptions se algo não correr bem?
            FilesOperation.storeSchool(school);
        }catch(DuplicatedDataException e){
            throw new ResponseStatusException(HttpStatus.CONFLICT, "Problems with "+e.getMessage(), e);
        }
    }

    @DeleteMapping( path = "{studentId}/subject/{subjectInitials}")
    public Enroll deleteLecture(@PathVariable Integer studentId, @PathVariable String subjectInitials) {
        Enroll enroll = getStudentSubject(studentId, subjectInitials);
        if(enroll != null){
            School school = FilesOperation.loadSchool();
            school.getStudentContainer().getEnrolls().remove(studentId, subjectInitials);
            FilesOperation.storeSchool(school);
            return enroll;
        }else{
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No Enroll with studentId" + studentId + "  and subject initials " + subjectInitials + " found");
        }
    }
}
